/**
 * Sync script to fetch job pins from Cotton Pins website and import them to Cotton City Plumbing database
 * This allows the two apps to share data without direct database connection
 */

import { createJobPin, getAllJobPins } from "./db";
import * as cheerio from "cheerio";

const COTTON_PINS_URL = "https://cottonpins-3rqvafpp.manus.space/jobs";

interface CottonPinJob {
  title: string;
  description: string;
  serviceType: string;
  address: string;
  photoUrl: string;
  date: string;
  location: string;
}

/**
 * Fetch and parse job pins from Cotton Pins website
 */
async function fetchCottonPinsJobs(): Promise<CottonPinJob[]> {
  console.log("[Sync] Fetching jobs from Cotton Pins...");
  
  const response = await fetch(COTTON_PINS_URL);
  if (!response.ok) {
    throw new Error(`Failed to fetch Cotton Pins: ${response.status}`);
  }
  
  const html = await response.text();
  const $ = cheerio.load(html);
  
  const jobs: CottonPinJob[] = [];
  
  // Parse job cards from the page
  $('a[href^="/jobs/"]').each((_: number, element: any) => {
    const $card = $(element);
    
    const title = $card.find('h3, .job-title').first().text().trim();
    const description = $card.find('p, .job-description').first().text().trim();
    const serviceType = $card.find('.service-type, .badge').first().text().trim();
    const location = $card.find('.location').text().trim() || "Lubbock";
    const address = $card.find('.address').text().trim() || location;
    const date = $card.find('.date, time').first().text().trim();
    
    // Extract image URL
    const img = $card.find('img').first();
    const photoUrl = img.attr('src') || img.attr('data-src') || "";
    
    if (title && description) {
      jobs.push({
        title,
        description,
        serviceType: serviceType || "General Plumbing",
        address,
        location,
        photoUrl,
        date
      });
    }
  });
  
  console.log(`[Sync] Found ${jobs.length} jobs from Cotton Pins`);
  return jobs;
}

/**
 * Sync Cotton Pins jobs to local database
 */
export async function syncCottonPinsToDatabase() {
  try {
    console.log("[Sync] Starting Cotton Pins sync...");
    
    // Fetch jobs from Cotton Pins
    const cottonPinsJobs = await fetchCottonPinsJobs();
    
    // Get existing jobs from local database
    const existingJobs = await getAllJobPins(true); // Include all statuses
    const existingTitles = new Set(existingJobs.map(j => j.title));
    
    // Import new jobs
    let importedCount = 0;
    for (const job of cottonPinsJobs) {
      // Skip if already exists
      if (existingTitles.has(job.title)) {
        continue;
      }
      
      try {
        await createJobPin({
          title: job.title,
          description: job.description,
          serviceType: job.serviceType,
          address: job.address,
          latitude: "33.5779", // Default Lubbock coordinates
          longitude: "-101.8552",
          photoUrl: job.photoUrl,
          photoKey: `cotton-pins-sync/${Date.now()}`,
          technicianName: null,
          customerReview: null,
          customerName: null,
          rating: null,
          altText: `${job.title} - ${job.description}`,
          createdBy: 1, // System user
          status: "approved", // Auto-approve synced pins
        });
        
        importedCount++;
        console.log(`[Sync] Imported: ${job.title}`);
      } catch (error) {
        console.error(`[Sync] Failed to import job "${job.title}":`, error);
      }
    }
    
    console.log(`[Sync] Completed! Imported ${importedCount} new jobs`);
    return { success: true, imported: importedCount, total: cottonPinsJobs.length };
    
  } catch (error) {
    console.error("[Sync] Sync failed:", error);
    throw error;
  }
}

// Run sync if executed directly
if (require.main === module) {
  syncCottonPinsToDatabase()
    .then(result => {
      console.log("[Sync] Result:", result);
      process.exit(0);
    })
    .catch(error => {
      console.error("[Sync] Error:", error);
      process.exit(1);
    });
}
