import { getAllJobPins } from "./db";

/**
 * Generate XML sitemap for SEO
 * Includes all static pages, job pins, service pages, and area pages
 */
export async function generateSitemap(baseUrl: string): Promise<string> {
  const currentDate = new Date().toISOString();

  // Static pages with priority and change frequency
  const staticPages = [
    { url: "/", priority: "1.0", changefreq: "daily" },
    { url: "/about", priority: "0.8", changefreq: "monthly" },
    { url: "/services", priority: "0.9", changefreq: "weekly" },
    { url: "/gallery", priority: "0.7", changefreq: "weekly" },
    { url: "/recent-jobs", priority: "0.9", changefreq: "daily" },
    { url: "/financing", priority: "0.7", changefreq: "monthly" },
    { url: "/contact", priority: "0.8", changefreq: "monthly" },
    { url: "/privacy-policy", priority: "0.3", changefreq: "yearly" },
    { url: "/terms-of-use", priority: "0.3", changefreq: "yearly" },
  ];

  // Service pages
  const servicePages = [
    "/services/water-heaters",
    "/services/drain-cleaning",
    "/services/emergency-plumbing",
    "/services/leak-detection",
    "/services/gas-lines",
    "/services/water-treatment",
    "/services/sewer-repair",
    "/services/repiping",
    "/services/commercial",
    "/services/fixtures",
    "/services/appliances",
    "/services/valves",
    "/services/slab-leaks",
  ].map(url => ({ url, priority: "0.8", changefreq: "monthly" }));

  // Area pages
  const areaPages = [
    "/areas/wolfforth",
    "/areas/slaton",
    "/areas/shallowater",
    "/areas/levelland",
    "/areas/idalou",
    "/areas/abernathy",
    "/areas/woodrow",
    "/areas/new-home",
    "/areas/new-deal",
    "/areas/smyer",
    "/areas/ropesville",
  ].map(url => ({ url, priority: "0.7", changefreq: "monthly" }));

  // Fetch all approved/featured job pins
  const jobPins = await getAllJobPins(false); // Only public pins

  // Job pin detail pages
  const jobPinPages = jobPins.map(job => ({
    url: `/recent-jobs/${job.id}`,
    priority: job.status === "featured" ? "0.9" : "0.7",
    changefreq: "weekly",
    lastmod: new Date(job.updatedAt).toISOString(),
  }));

  // Combine all pages
  const allPages = [
    ...staticPages.map(page => ({ ...page, lastmod: currentDate })),
    ...servicePages.map(page => ({ ...page, lastmod: currentDate })),
    ...areaPages.map(page => ({ ...page, lastmod: currentDate })),
    ...jobPinPages,
  ];

  // Generate XML
  const xml = `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">
${allPages
  .map(
    page => `  <url>
    <loc>${baseUrl}${page.url}</loc>
    <lastmod>${page.lastmod}</lastmod>
    <changefreq>${page.changefreq}</changefreq>
    <priority>${page.priority}</priority>
  </url>`
  )
  .join("\n")}
</urlset>`;

  return xml;
}
