/**
 * Fetch job pins from Cotton Pins website via TRPC API
 * This allows Cotton City Plumbing website to display pins uploaded via the Cotton Pins app
 */

const COTTON_PINS_API_URL = "https://cottonpins-3rqvafpp.manus.space/api/trpc/dataPins.list";

export interface CottonPin {
  id: number;
  title: string;
  description: string;
  serviceType: string;
  address: string;
  latitude: string;
  longitude: string;
  photoUrl: string;
  date: string;
  location: string;
  city?: string;
  neighborhood?: string;
}

/**
 * Fetch all job pins from Cotton Pins website via TRPC API
 */
export async function fetchCottonPinsJobs(): Promise<CottonPin[]> {
  try {
    console.log("[CottonPins] Fetching jobs from Cotton Pins API...");
    
    const response = await fetch(COTTON_PINS_API_URL);
    if (!response.ok) {
      throw new Error(`Failed to fetch: ${response.status}`);
    }
    
    const data = await response.json();
    
    // Extract the result data from TRPC response format
    const pins = data?.result?.data?.json || data?.result?.data || [];
    
    if (!Array.isArray(pins)) {
      console.error("[CottonPins] Unexpected response format:", data);
      return [];
    }
    
    console.log(`[CottonPins] Found ${pins.length} jobs from Cotton Pins`);
    
    // Transform Cotton Pins data to our format
    return pins.map((pin: any) => {
      let address = pin.address || "";
      const city = pin.city || "Lubbock";
      const neighborhood = pin.neighborhood;
      
      // Clean up address formatting
      address = address.trim();
      
      // If address doesn't contain city/state/zip, append them
      const hasStateZip = /TX\s+\d{5}/.test(address);
      const hasCity = address.toLowerCase().includes(city.toLowerCase());
      
      if (address && !hasStateZip) {
        // Address exists but missing city/state/zip
        if (!hasCity) {
          address = `${address}, ${city}, TX`;
        } else {
          address = `${address}, TX`;
        }
      } else if (!address) {
        // No address at all
        address = neighborhood ? `${neighborhood}, ${city}, TX` : `${city}, TX`;
      }
      
      return {
        id: pin.id,
        title: pin.title || pin.jobTitle || "Untitled Job",
        description: pin.description || pin.notes || "",
        serviceType: pin.serviceType || "General Plumbing",
        address,
        latitude: pin.latitude?.toString() || "33.5779",
        longitude: pin.longitude?.toString() || "-101.8552",
        photoUrl: pin.photoUrl || pin.imageUrl || "",
        date: pin.jobDate || pin.createdAt || new Date().toISOString(),
        location: pin.location || city,
        city,
        neighborhood,
      };
    });
    
  } catch (error) {
    console.error("[CottonPins] Failed to fetch jobs:", error);
    return [];
  }
}
