import { eq, sql, or } from "drizzle-orm";
import { drizzle } from "drizzle-orm/mysql2";
import { InsertUser, users, InsertJobPin, jobPins } from "../drizzle/schema";
import { ENV } from './_core/env';

let _db: ReturnType<typeof drizzle> | null = null;

// Lazily create the drizzle instance so local tooling can run without a DB.
export async function getDb() {
  if (!_db && process.env.DATABASE_URL) {
    try {
      _db = drizzle(process.env.DATABASE_URL);
    } catch (error) {
      console.warn("[Database] Failed to connect:", error);
      _db = null;
    }
  }
  return _db;
}

export async function upsertUser(user: InsertUser): Promise<void> {
  if (!user.openId) {
    throw new Error("User openId is required for upsert");
  }

  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot upsert user: database not available");
    return;
  }

  try {
    const values: InsertUser = {
      openId: user.openId,
    };
    const updateSet: Record<string, unknown> = {};

    const textFields = ["name", "email", "loginMethod"] as const;
    type TextField = (typeof textFields)[number];

    const assignNullable = (field: TextField) => {
      const value = user[field];
      if (value === undefined) return;
      const normalized = value ?? null;
      values[field] = normalized;
      updateSet[field] = normalized;
    };

    textFields.forEach(assignNullable);

    if (user.lastSignedIn !== undefined) {
      values.lastSignedIn = user.lastSignedIn;
      updateSet.lastSignedIn = user.lastSignedIn;
    }
    if (user.role !== undefined) {
      values.role = user.role;
      updateSet.role = user.role;
    } else if (user.openId === ENV.ownerOpenId) {
      values.role = 'admin';
      updateSet.role = 'admin';
    }

    if (!values.lastSignedIn) {
      values.lastSignedIn = new Date();
    }

    if (Object.keys(updateSet).length === 0) {
      updateSet.lastSignedIn = new Date();
    }

    await db.insert(users).values(values).onDuplicateKeyUpdate({
      set: updateSet,
    });
  } catch (error) {
    console.error("[Database] Failed to upsert user:", error);
    throw error;
  }
}

export async function getUserByOpenId(openId: string) {
  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot get user: database not available");
    return undefined;
  }

  const result = await db.select().from(users).where(eq(users.openId, openId)).limit(1);

  return result.length > 0 ? result[0] : undefined;
}

/**
 * Create a new job pin
 */
export async function createJobPin(jobPin: InsertJobPin) {
  const db = await getDb();
  if (!db) {
    throw new Error("Database not available");
  }

  const result = await db.insert(jobPins).values(jobPin);
  return result;
}

/**
 * Get all job pins, ordered by most recent first
 * @param includeHidden - If true, includes hidden pins (for admin view)
 */
export async function getAllJobPins(includeHidden = false) {
  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot get job pins: database not available");
    return [];
  }

  let result;
  
  if (!includeHidden) {
    // Only show approved and featured pins to public
    result = await db.select().from(jobPins)
      .where(
        or(
          eq(jobPins.status, "approved"),
          eq(jobPins.status, "featured")
        )
      )
      .orderBy(jobPins.createdAt);
  } else {
    // Show all pins for admin
    result = await db.select().from(jobPins).orderBy(jobPins.createdAt);
  }
  return result;
}

/**
 * Update job pin status (for admin moderation)
 */
export async function updateJobPinStatus(id: number, status: "pending" | "approved" | "featured" | "hidden") {
  const db = await getDb();
  if (!db) {
    throw new Error("Database not available");
  }

  await db.update(jobPins).set({ status }).where(eq(jobPins.id, id));
}

/**
 * Delete a job pin (for admin)
 */
export async function deleteJobPin(id: number) {
  const db = await getDb();
  if (!db) {
    throw new Error("Database not available");
  }

  await db.delete(jobPins).where(eq(jobPins.id, id));
}

/**
 * Get a single job pin by ID
 */
export async function getJobPinById(id: number) {
  const db = await getDb();
  if (!db) {
    console.warn("[Database] Cannot get job pin: database not available");
    return undefined;
  }

  const result = await db.select().from(jobPins).where(eq(jobPins.id, id)).limit(1);
  return result.length > 0 ? result[0] : undefined;
}
