import { ENV } from "./_core/env";

/**
 * Generate descriptive alt text for job photos using AI vision model
 * This improves SEO and accessibility by providing accurate image descriptions
 */
export async function generateAltText(
  imageUrl: string,
  jobTitle: string,
  serviceType: string,
  address: string
): Promise<string> {
  try {
    const apiKey = ENV.forgeApiKey;
    const apiUrl = ENV.forgeApiUrl;

    if (!apiKey || !apiUrl) {
      console.warn("[Alt Text] Forge API not configured, using fallback");
      return generateFallbackAltText(jobTitle, serviceType, address);
    }

    // Call Manus Forge API for image analysis
    const response = await fetch(`${apiUrl}/v1/chat/completions`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Authorization: `Bearer ${apiKey}`,
      },
      body: JSON.stringify({
        model: "gpt-4o-mini",
        messages: [
          {
            role: "system",
            content: `You are an SEO expert writing alt text for plumbing job photos. Generate concise, descriptive alt text (max 125 characters) that:
1. Describes what's visible in the image
2. Includes the service type and location when relevant
3. Uses natural language optimized for image search
4. Focuses on the plumbing work shown

Format: "[Service] - [What's shown] in [Location context]"
Example: "Tankless water heater installation in residential garage, Lubbock TX"`,
          },
          {
            role: "user",
            content: [
              {
                type: "text",
                text: `Generate alt text for this plumbing job photo. Context: ${jobTitle}, Service: ${serviceType}, Location: ${address}`,
              },
              {
                type: "image_url",
                image_url: {
                  url: imageUrl,
                },
              },
            ],
          },
        ],
        max_tokens: 100,
        temperature: 0.3,
      }),
    });

    if (!response.ok) {
      console.error("[Alt Text] API error:", response.status, response.statusText);
      return generateFallbackAltText(jobTitle, serviceType, address);
    }

    const data = await response.json();
    const altText = data.choices?.[0]?.message?.content?.trim();

    if (!altText) {
      console.warn("[Alt Text] Empty response from API, using fallback");
      return generateFallbackAltText(jobTitle, serviceType, address);
    }

    // Ensure alt text is within reasonable length (125 chars is Google's recommendation)
    const truncatedAltText = altText.length > 125 ? altText.substring(0, 122) + "..." : altText;

    console.log(`[Alt Text] Generated: "${truncatedAltText}"`);
    return truncatedAltText;
  } catch (error) {
    console.error("[Alt Text] Error generating alt text:", error);
    return generateFallbackAltText(jobTitle, serviceType, address);
  }
}

/**
 * Fallback alt text generation when AI is unavailable
 * Creates SEO-friendly descriptions from job metadata
 */
function generateFallbackAltText(jobTitle: string, serviceType: string, address: string): string {
  // Extract city from address (e.g., "123 Main St, Lubbock, TX 79401" -> "Lubbock")
  const city = address.split(",")[1]?.trim() || "Lubbock";

  // Create descriptive alt text from available data
  const altText = `${serviceType} - ${jobTitle} in ${city}, TX`;

  // Truncate if needed
  return altText.length > 125 ? altText.substring(0, 122) + "..." : altText;
}
