import mysql from 'mysql2/promise';

const DATABASE_URL = process.env.DATABASE_URL;

// The 16 correct jobs from the screenshots
const correctJobs = [
  { title: "Water Line Repair", date: "12/3/2025" },
  { title: "Sewer Line Replacement", date: "12/3/2025" },
  { title: "Sewer Line Repair", date: "12/3/2025" },
  { title: "Water Heater Installation", date: "12/1/2025" },
  { title: "Hose Bib Replacement", date: "12/1/2025" },
  { title: "Tankless Water Heater Installation", date: "12/1/2025" },
  { title: "Bathtub Installation", date: "12/1/2025" },
  { title: "P-Trap Replacement", date: "12/1/2025" },
  { title: "Naturalsof Water Conditioner Installation in The Ridge", date: "11/27/2025" },
  { title: "Water Heater Installation in The Ridge", date: "11/27/2025" },
];

async function findDuplicates() {
  const connection = await mysql.createConnection(DATABASE_URL);
  
  try {
    const [rows] = await connection.execute(
      'SELECT id, title, address, latitude, longitude, photoUrl, createdAt FROM jobPins ORDER BY createdAt DESC, title'
    );
    
    console.log(`\n${'='.repeat(120)}`);
    console.log(`ALL ${rows.length} JOBS IN DATABASE`);
    console.log('='.repeat(120));
    
    // Group by title to find duplicates
    const jobsByTitle = {};
    rows.forEach(job => {
      const key = job.title.trim();
      if (!jobsByTitle[key]) {
        jobsByTitle[key] = [];
      }
      jobsByTitle[key].push(job);
    });
    
    console.log('\n\nJOBS GROUPED BY TITLE:\n');
    Object.keys(jobsByTitle).sort().forEach(title => {
      const jobs = jobsByTitle[title];
      console.log(`\n"${title}" - ${jobs.length} entries:`);
      jobs.forEach(job => {
        const date = new Date(job.createdAt).toLocaleDateString();
        console.log(`  ID ${job.id}: ${job.address} | GPS: ${job.latitude}, ${job.longitude} | ${date}`);
        console.log(`    Photo: ${job.photoUrl?.split('/').pop()}`);
      });
    });
    
    // Find potential duplicates (same title, same GPS)
    console.log(`\n\n${'='.repeat(120)}`);
    console.log('DUPLICATE CANDIDATES (same title + same GPS):');
    console.log('='.repeat(120));
    
    const duplicatesToDelete = [];
    
    Object.keys(jobsByTitle).forEach(title => {
      const jobs = jobsByTitle[title];
      if (jobs.length > 1) {
        // Check for GPS duplicates
        const gpsGroups = {};
        jobs.forEach(job => {
          const gpsKey = `${job.latitude.toFixed(4)},${job.longitude.toFixed(4)}`;
          if (!gpsGroups[gpsKey]) {
            gpsGroups[gpsKey] = [];
          }
          gpsGroups[gpsKey].push(job);
        });
        
        Object.keys(gpsGroups).forEach(gpsKey => {
          const group = gpsGroups[gpsKey];
          if (group.length > 1) {
            console.log(`\n"${title}" at GPS ${gpsKey}:`);
            // Sort by created date, keep oldest
            group.sort((a, b) => new Date(a.createdAt) - new Date(b.createdAt));
            console.log(`  KEEP: ID ${group[0].id} (${group[0].address})`);
            for (let i = 1; i < group.length; i++) {
              console.log(`  DELETE: ID ${group[i].id} (${group[i].address})`);
              duplicatesToDelete.push(group[i].id);
            }
          }
        });
      }
    });
    
    if (duplicatesToDelete.length > 0) {
      console.log(`\n\n${'='.repeat(120)}`);
      console.log('FINAL DELETION COMMAND:');
      console.log('='.repeat(120));
      console.log(`\nTotal duplicates to delete: ${duplicatesToDelete.length}`);
      console.log(`Jobs after deletion: ${rows.length - duplicatesToDelete.length}\n`);
      console.log(`DELETE FROM jobPins WHERE id IN (${duplicatesToDelete.join(', ')});`);
    } else {
      console.log('\n\nNo exact duplicates found by title + GPS matching.');
      console.log('Manual review needed to identify the 5 duplicates.');
    }
    
  } finally {
    await connection.end();
  }
}

findDuplicates().catch(console.error);
