import { int, mysqlEnum, mysqlTable, text, timestamp, varchar } from "drizzle-orm/mysql-core";

/**
 * Core user table backing auth flow.
 * Extend this file with additional tables as your product grows.
 * Columns use camelCase to match both database fields and generated types.
 */
export const users = mysqlTable("users", {
  /**
   * Surrogate primary key. Auto-incremented numeric value managed by the database.
   * Use this for relations between tables.
   */
  id: int("id").autoincrement().primaryKey(),
  /** Manus OAuth identifier (openId) returned from the OAuth callback. Unique per user. */
  openId: varchar("openId", { length: 64 }).notNull().unique(),
  name: text("name"),
  email: varchar("email", { length: 320 }),
  loginMethod: varchar("loginMethod", { length: 64 }),
  role: mysqlEnum("role", ["user", "admin"]).default("user").notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
  lastSignedIn: timestamp("lastSignedIn").defaultNow().notNull(),
});

export type User = typeof users.$inferSelect;
export type InsertUser = typeof users.$inferInsert;

/**
 * Job pins table for the Data Pins feature.
 * Stores geo-tagged plumbing jobs submitted by technicians.
 */
export const jobPins = mysqlTable("jobPins", {
  id: int("id").autoincrement().primaryKey(),
  /** Job title (e.g., "Water Heater Replacement - Tech Terrace") */
  title: varchar("title", { length: 255 }).notNull(),
  /** Job description (2-3 sentences about what was done) */
  description: text("description").notNull(),
  /** Service type (e.g., "Water Heaters", "Drain Cleaning", etc.) */
  serviceType: varchar("serviceType", { length: 100 }).notNull(),
  /** Full address of the job location */
  address: varchar("address", { length: 500 }).notNull(),
  /** GPS latitude coordinate */
  latitude: varchar("latitude", { length: 50 }).notNull(),
  /** GPS longitude coordinate */
  longitude: varchar("longitude", { length: 50 }).notNull(),
  /** Photo URL (stored in S3) */
  photoUrl: varchar("photoUrl", { length: 1000 }).notNull(),
  /** S3 file key for the photo */
  photoKey: varchar("photoKey", { length: 500 }).notNull(),
  /** Technician name who completed the job */
  technicianName: varchar("technicianName", { length: 255 }),
  /** Customer review/testimonial (optional) */
  customerReview: text("customerReview"),
  /** Customer name (optional, for attribution) */
  customerName: varchar("customerName", { length: 255 }),
  /** Star rating (1-5) */
  rating: int("rating"),
  /** AI-generated alt text for the photo (for SEO and accessibility) */
  altText: text("altText"),
  /** User ID who created this pin (references users table) */
  createdBy: int("createdBy").notNull(),
  /** Moderation status: pending (default), approved, featured, hidden */
  status: mysqlEnum("status", ["pending", "approved", "featured", "hidden"]).default("pending").notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type JobPin = typeof jobPins.$inferSelect;
export type InsertJobPin = typeof jobPins.$inferInsert;