import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { MapPin, Calendar, User, Loader2 } from "lucide-react";
import { Link } from "wouter";
import { MapView } from "@/components/Map";
import { trpc } from "@/lib/trpc";
import { JobPinSchema } from "@/components/JobPinSchema";

export default function RecentJobs() {
  // Fetch job pins from the API
  const { data: jobPins, isLoading, error } = trpc.jobPins.list.useQuery();

  return (
    <Layout>
      <SEO 
        title="Recent Plumbing Jobs in Lubbock | Cotton City Plumbing"
        description="See our recent plumbing work in Lubbock, Wolfforth, and surrounding areas. Real jobs, real photos, and real results from Cotton City Plumbing."
        canonical="/recent-jobs"
      />

      <section className="bg-black text-white py-16">
        <div className="container text-center">
          <h1 className="font-league-gothic text-5xl md:text-7xl mb-4 uppercase tracking-wide">Check-Ins</h1>
          <p className="text-xl text-gray-300 max-w-2xl mx-auto">
            See where our technicians are working near you. We're proud to serve the entire Lubbock community!
          </p>
        </div>
      </section>

      <section className="py-12 bg-white">
        <div className="container max-w-7xl">
          
          {isLoading && (
            <div className="flex items-center justify-center py-24">
              <Loader2 className="h-12 w-12 animate-spin text-red-600" />
            </div>
          )}

          {error && (
            <div className="text-center py-24">
              <p className="text-red-600 font-bold text-xl">Error loading jobs</p>
              <p className="text-gray-500 mt-2">Please try again later</p>
            </div>
          )}

          {jobPins && jobPins.length === 0 && (
            <div className="text-center py-24">
              <p className="text-gray-600 font-bold text-xl">No recent jobs yet</p>
              <p className="text-gray-500 mt-2">Check back soon to see our latest work!</p>
            </div>
          )}

          {/* 2-Column Grid Layout */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {jobPins && jobPins.map((job) => (
              <Link key={job.id} href={`/recent-jobs/${job.id}`}>
                <JobPinSchema jobPin={job} />
                <Card className="hover:shadow-2xl transition-all duration-300 cursor-pointer overflow-hidden border-2 hover:border-red-600">
                  <CardContent className="p-0">
                    {/* Technician & Date Header */}
                    <div className="bg-white p-4 border-b">
                      <div className="flex items-center justify-between">
                        {job.technicianName && (
                          <div className="flex items-center gap-2">
                            <User className="h-4 w-4 text-gray-500" />
                            <span className="font-bold text-sm text-zinc-900">{job.technicianName}</span>
                          </div>
                        )}
                        <span className="text-xs text-gray-500 flex items-center gap-1">
                          <Calendar className="h-3 w-3" /> 
                          {new Date(job.createdAt).toLocaleDateString('en-US', { 
                            month: 'short', 
                            day: 'numeric', 
                            year: 'numeric' 
                          })}
                        </span>
                      </div>
                    </div>

                    {/* Title */}
                    <div className="px-4 pt-4">
                      <h3 className="font-bold text-xl text-zinc-900 mb-3">{job.title}</h3>
                    </div>

                    {/* Map and Photo Side-by-Side */}
                    <div className="grid grid-cols-2 gap-0">
                      {/* Embedded Interactive Map */}
                      <div className="h-64 border-r">
                        <MapView 
                          className="w-full h-full"
                          initialCenter={{ lat: parseFloat(job.latitude), lng: parseFloat(job.longitude) }}
                          initialZoom={15}
                          onMapReady={(map) => {
                            new google.maps.Marker({
                              position: { lat: parseFloat(job.latitude), lng: parseFloat(job.longitude) },
                              map: map,
                              title: job.title,
                            });
                          }}
                        />
                      </div>

                      {/* Job Photo */}
                      <div className="h-64 overflow-hidden">
                        <img 
                          src={job.photoUrl} 
                          alt={job.altText || job.title} 
                          className="w-full h-full object-cover hover:scale-110 transition-transform duration-700"
                          loading="lazy"
                        />
                      </div>
                    </div>

                    {/* Description & Address */}
                    <div className="p-4">
                      <p className="text-sm text-gray-600 mb-3">
                        {job.description}
                      </p>
                      <div className="flex items-center gap-1 text-xs font-bold text-red-600">
                        <MapPin className="h-3 w-3" /> {job.address}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
        </div>
      </section>

      <section className="py-16 bg-gray-100 text-center">
        <div className="container">
          <h2 className="font-league-gothic text-4xl text-zinc-900 mb-6 uppercase">Need Service in Your Area?</h2>
          <p className="text-lg text-gray-600 mb-8 max-w-2xl mx-auto">
            Our team is always on the move. Contact us today to schedule your service!
          </p>
          <div className="flex flex-col sm:flex-row justify-center gap-4">
            <a href="tel:8064414110">
              <Button size="lg" className="bg-red-600 hover:bg-red-700 text-white font-bold text-lg px-10 py-6 shadow-xl uppercase tracking-wider">
                Call (806) 441-4110
              </Button>
            </a>
            <Link href="/contact">
              <Button size="lg" className="bg-primary hover:bg-primary/90 text-white font-bold text-lg px-10 py-6 uppercase tracking-wider shadow-xl">
                Book Online Now
              </Button>
            </Link>
          </div>
        </div>
      </section>
    </Layout>
  );
}
