import { useAuth } from "@/_core/hooks/useAuth";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Check, ChevronRight, Clock, Droplets, Flame, Hammer, Heart, MapPin, Phone, Shield, Star, Wrench } from "lucide-react";
import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import { Link } from "wouter";
import GoogleReviewsWidget from "@/components/GoogleReviewsWidget";

export default function Home() {
  // The userAuth hooks provides authentication state
  // To implement login/logout functionality, simply call logout() or redirect to getLoginUrl()
  let { user, loading, error, isAuthenticated, logout } = useAuth();

  return (
    <Layout>
      <SEO 
        title="Best Plumber in Lubbock, TX | Same-Day Emergency Service"
        description="Cotton City Plumbing offers trusted, faith-based plumbing services in Lubbock, TX. From emergency repairs to water heaters and drain cleaning, we do it all. Call (806) 441-4110."
        canonical="/"
      />
      {/* Financing Banner */}
      <div className="bg-primary text-white py-3 text-center font-bold text-lg md:text-xl">
        <div className="container flex items-center justify-center gap-2">
          <Shield className="h-5 w-5" />
          <span>Ask About Financing! 0% APR Options Available for Qualified Buyers.</span>
          <Link href="/financing" className="underline hover:no-underline ml-2">Learn More →</Link>
        </div>
      </div>

      {/* Hero Section */}
      <section className="relative min-h-[80vh] flex items-center justify-center overflow-hidden bg-black">
        {/* Solid Black Background */}
        <div className="absolute inset-0 z-0 bg-black"></div>

        <div className="container relative z-10 py-20">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            {/* Left Column: Text Content */}
            <div className="space-y-6 animate-in slide-in-from-left duration-700">
            {/* Awards Badges - Universal Improvement #2 */}
            <div className="flex flex-wrap items-center gap-6 mb-6">
              <img 
                src="/images/community-choice-2025.png" 
                alt="2025 Official Community's Choice Awards - Best of Lubbock" 
                className="h-24 w-auto drop-shadow-lg hover:scale-105 transition-transform duration-300"
              />
              <img 
                src="/images/nextdoor-fave-2025.jpg" 
                alt="2025 Nextdoor Neighborhood Faves Winner" 
                className="h-24 w-auto drop-shadow-lg hover:scale-105 transition-transform duration-300 rounded-full border-2 border-white/20"
              />
              <img 
                src="/images/businessrate-best-of-2025.png" 
                alt="Best of BusinessRate 2025 - Plumber in Lubbock, TX" 
                className="h-24 w-auto drop-shadow-lg hover:scale-105 transition-transform duration-300"
              />
              <div className="bg-white/10 backdrop-blur-sm border border-white/20 rounded-none px-4 py-2 text-sm font-bold text-white flex items-center gap-2 uppercase tracking-wider h-fit">
                <Shield className="h-4 w-4 text-primary fill-primary" />
                Licensed & Insured M-44446
              </div>
            </div>

            <h1 className="font-heading font-extrabold text-5xl md:text-7xl lg:text-8xl text-white leading-none drop-shadow-lg uppercase italic">
              We Make Sure Everything <span className="text-primary">Runs Through Lubbock</span>
            </h1>
            
            <p className="text-xl md:text-2xl text-gray-300 font-light max-w-2xl leading-relaxed">
              Lubbock's trusted, faith-based plumbing experts. From emergency repairs to new installations, we serve our neighbors with honesty, integrity, and excellence.
            </p>

            <div className="flex flex-col sm:flex-row gap-4 pt-4">
              <a href="tel:8064414110" className="w-full sm:w-auto">
                <Button size="lg" className="w-full sm:w-auto bg-primary hover:bg-primary/90 text-white font-heading font-bold text-2xl px-8 py-8 shadow-xl hover:shadow-2xl hover:-translate-y-1 transition-all uppercase rounded-none border-2 border-primary">
                  <Phone className="mr-2 h-6 w-6" /> Call (806) 441-4110
                </Button>
              </a>
              <Link href="/contact">
                <Button size="lg" variant="outline" className="w-full sm:w-auto border-2 border-white text-white hover:bg-white hover:text-black font-heading font-bold text-2xl px-8 py-8 bg-transparent backdrop-blur-sm uppercase rounded-none">
                  Book Online
                </Button>
              </Link>
            </div>
          </div>

          {/* Right Column: Mascot Logo */}
          <div className="hidden lg:flex items-center justify-center animate-in slide-in-from-right duration-700">
            <img 
              src="/images/squarespace-hero-section.webp" 
              alt="Cotton City Plumbing mascot with red cowboy hat" 
              className="w-full max-w-2xl drop-shadow-2xl scale-110"
              loading="eager"
              width="750"
              height="422"
            />
          </div>
        </div>
        </div>
        
        {/* Wave Divider */}
        <div className="absolute bottom-0 left-0 right-0 h-16 bg-background" style={{ clipPath: "polygon(0 100%, 100% 100%, 100% 0, 0 100%)" }}></div>
      </section>

      {/* Introduction & Team Section - Critical Gap #1 */}
      <section className="py-20 bg-black">
        <div className="container">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div className="space-y-6">
              <div className="inline-block">
                <h2 className="font-heading font-bold text-4xl md:text-5xl text-white mb-2 uppercase italic">Meet Your <span className="text-primary">Cotton City Team</span></h2>
                <div className="h-1 w-20 bg-primary rounded-none"></div>
              </div>
              
              <p className="text-lg text-gray-400 leading-relaxed">
                As a faith-based, locally owned and operated company, we believe in more than just fixing pipes. We believe in serving people. Founded on Christian principles, our mission is to bring honesty, integrity, and top-quality craftsmanship to every home in Lubbock.
              </p>
              
              <p className="text-lg text-gray-400 leading-relaxed">
                Led by a Master Plumber with <span className="font-bold text-primary">30+ years of industry experience</span>, our team uses state-of-the-art camera inspection technology to diagnose problems accurately—saving you time, money, and stress.
              </p>

              <div className="grid grid-cols-2 gap-4 pt-4">
                {[
                  "Faith-Based Values",
                  "Transparent Pricing",
                  "Advanced Technology",
                  "Respect For Your Home"
                ].map((item) => (
                  <div key={item} className="flex items-center gap-2">
                    <div className="bg-primary/10 p-1 rounded-none">
                      <Check className="h-4 w-4 text-primary" />
                    </div>
                    <span className="font-medium text-gray-300">{item}</span>
                  </div>
                ))}
              </div>
              
              <Link href="/about">
                <Button variant="link" className="text-primary font-bold text-lg p-0 h-auto mt-4 group uppercase tracking-wide hover:text-white transition-colors">
                  Read Our Story <ChevronRight className="h-4 w-4 ml-1 group-hover:translate-x-1 transition-transform" />
                </Button>
              </Link>
            </div>

            <div className="relative">
              {/* Main Team Photo */}
              <div className="relative z-10 rounded-none overflow-hidden shadow-2xl border-4 border-white/10 transform hover:scale-[1.02] transition-transform duration-500">
                <img 
                  src="/images/team-homepage.png" 
                  alt="Cotton City Plumbing licensed team of plumbers in Lubbock TX" 
                  className="w-full h-auto object-cover grayscale hover:grayscale-0 transition-all duration-500"
                  loading="lazy"
                  width="800"
                  height="600"
                />
                <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/90 to-transparent p-6">
                  <p className="text-white font-heading font-bold text-2xl uppercase italic">Licensed, Uniformed & Professional</p>
                  <p className="text-primary font-bold text-sm uppercase tracking-wider">Ready to serve your home today</p>
                </div>
              </div>
              
              {/* Decorative Element */}
              <div className="absolute -top-6 -right-6 w-32 h-32 border-2 border-primary/30 -z-0"></div>
              <div className="absolute -bottom-6 -left-6 w-40 h-40 bg-primary/10 -z-0"></div>
            </div>
          </div>
        </div>
      </section>

      {/* Expanded Services Section - Critical Gap #2 */}
      <section className="py-20 bg-[#111111] relative">
        <div className="container">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <h2 className="font-heading font-bold text-4xl md:text-5xl text-white mb-4 uppercase italic">Comprehensive <span className="text-primary">Plumbing Solutions</span></h2>
            <p className="text-lg text-gray-400">
              From minor leaks to major installations, we have the expertise and technology to handle it all.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {[
              {
                title: "Emergency Plumbing",
                icon: <Clock className="h-8 w-8 text-primary" />,
                desc: "Same-day emergency response for bursts, leaks, and backups during business hours.",
                link: "/services/emergency-plumbing"
              },
              {
                title: "Water Heaters",
                icon: <Flame className="h-8 w-8 text-primary" />,
                desc: "Installation and repair of tank and tankless systems for endless hot water.",
                link: "/services/water-heaters"
              },
              {
                title: "Drain Cleaning",
                icon: <Droplets className="h-8 w-8 text-primary" />,
                desc: "Professional clearing of stubborn clogs using safe, mechanical methods.",
                link: "/services/drain-cleaning"
              },
              {
                title: "Leak Detection",
                icon: <MapPin className="h-8 w-8 text-primary" />,
                desc: "Pinpoint hidden leaks behind walls or under slabs with camera tech.",
                link: "/services/leak-detection"
              },
              {
                title: "Gas Lines",
                icon: <Flame className="h-8 w-8 text-primary" />,
                desc: "Safe, certified installation and repair of gas lines for your home.",
                link: "/services/gas-lines"
              },
              {
                title: "Water Treatment",
                icon: <Droplets className="h-8 w-8 text-primary" />,
                desc: "Softeners and filtration systems for clean, healthy water throughout your home.",
                link: "/services/water-treatment"
              },
              {
                title: "Slab Leaks",
                icon: <Hammer className="h-8 w-8 text-primary" />,
                desc: "Specialized detection and repair to protect your home's foundation.",
                link: "/services/slab-leaks"
              },
              {
                title: "Commercial",
                icon: <Wrench className="h-8 w-8 text-primary" />,
                desc: "Reliable plumbing services for Lubbock businesses and properties.",
                link: "/services/commercial"
              }
            ].map((service, index) => (
              <Link key={index} href={service.link} className="block h-full">
                <Card className="h-full hover:shadow-xl transition-all duration-300 hover:-translate-y-1 border border-white/10 hover:border-primary group cursor-pointer bg-black">
                  <CardContent className="p-6 flex flex-col h-full">
                    <div className="bg-white/5 p-4 rounded-none w-16 h-16 flex items-center justify-center mb-4 group-hover:bg-primary/20 transition-colors">
                      {service.icon}
                    </div>
                    <h3 className="font-heading font-bold text-2xl text-white mb-2 group-hover:text-primary transition-colors uppercase italic">{service.title}</h3>
                    <p className="text-muted-foreground text-sm mb-4 flex-grow">{service.desc}</p>
                    <div className="flex items-center text-primary font-bold text-sm mt-auto group-hover:translate-x-1 transition-transform">
                      Learn More <ChevronRight className="h-4 w-4 ml-1" />
                    </div>
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
          
          <div className="text-center mt-12">
            <Link href="/services">
              <Button size="lg" className="bg-primary hover:bg-primary/90 text-white font-bold px-8">
                View All Services
              </Button>
            </Link>
          </div>
        </div>
      </section>

            {/* "Our Work" Gallery Preview - Critical Gap #1 */}
      <section className="py-20 bg-primary text-white overflow-hidden">
        <div className="container">
          <div className="flex flex-col md:flex-row justify-between items-end mb-12 gap-6">
            <div className="max-w-2xl">
              <h2 className="font-heading font-bold text-3xl md:text-4xl mb-4">See Our Quality Work</h2>
              <p className="text-primary-foreground/80 text-lg">
                We take pride in our craftsmanship. Here are a few examples of recent projects completed right here in Lubbock.
              </p>
            </div>
            <Link href="/gallery">
              <Button variant="outline" className="border-white text-white hover:bg-white hover:text-primary font-bold">
                View Full Gallery
              </Button>
            </Link>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {[
              {
                image: "/images/work-1.png",
                title: "Kitchen Faucet Installation",
                location: "Lubbock, TX"
              },
              {
                image: "/images/work-2.png",
                title: "Bathroom Faucet Installation",
                location: "Wolfforth, TX"
              },
              {
                image: "/images/work-3.png",
                title: "Tub/Shower Faucet Installation",
                location: "Tech Terrace, Lubbock"
              }
            ].map((project, index) => (
              <div key={index} className="group relative overflow-hidden rounded-xl aspect-[4/3] cursor-pointer">
                <img 
                  src={project.image} 
                  alt={project.title} 
                  className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-110"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent opacity-80 group-hover:opacity-100 transition-opacity"></div>
                <div className="absolute bottom-0 left-0 p-6 translate-y-2 group-hover:translate-y-0 transition-transform duration-300">
                  <h3 className="font-heading font-bold text-xl text-white mb-1">{project.title}</h3>
                  <div className="flex items-center text-secondary text-sm font-medium">
                    <MapPin className="h-3 w-3 mr-1" /> {project.location}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Why Choose Us - Expanded Trust Signals */}
      <section className="py-20 bg-background">
        <div className="container">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <h2 className="font-heading font-bold text-3xl md:text-4xl text-primary mb-4">Why Lubbock Chooses Cotton City</h2>
            <p className="text-lg text-muted-foreground">
              We're not just another plumbing company. We're your neighbors, dedicated to raising the standard of service in West Texas.
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                title: "Faith-Based Integrity",
                desc: "Our business is built on Christian principles. We treat every customer with honesty, fairness, and respect.",
                icon: <Star className="h-6 w-6" />
              },
              {
                title: "Upfront Pricing",
                desc: "No surprise fees or hidden charges. You'll know the exact price before we start any work.",
                icon: <Shield className="h-6 w-6" />
              },
              {
                title: "Respect For Your Home",
                desc: "We wear shoe covers, use drop cloths, and leave your home cleaner than we found it.",
                icon: <Check className="h-6 w-6" />
              },
              {
                title: "Advanced Technology",
                desc: "We use the latest camera inspection and leak detection tech to solve problems faster and less invasively.",
                icon: <Wrench className="h-6 w-6" />
              },
              {
                title: "Locally Owned",
                desc: "Born and raised in Lubbock. We're not a franchise; we're your neighbors serving neighbors.",
                icon: <MapPin className="h-6 w-6" />
              },
              {
                title: "Community Focused",
                desc: (
                  <>
                    Proud sponsor of{" "}
                    <a 
                      href="https://foreverlychanged.com" 
                      target="_blank" 
                      rel="noopener noreferrer"
                      className="text-primary hover:underline font-semibold"
                    >
                      ForEverly Changed
                    </a>
                    {" "}and active supporter of our Lubbock community.
                  </>
                ),
                icon: <Heart className="h-6 w-6" />
              },
              {
                title: "Satisfaction Guaranteed",
                desc: "We stand behind our work. If you're not happy, we'll make it right. That's the Cotton City promise.",
                icon: <Star className="h-6 w-6" />
              }
            ].map((feature, index) => (
              <div key={index} className="flex gap-4 p-6 rounded-xl bg-muted/30 hover:bg-white hover:shadow-lg transition-all border border-transparent hover:border-primary/10">
                <div className="bg-secondary/10 p-3 rounded-full h-fit text-secondary">
                  {feature.icon}
                </div>
                <div>
                  <h3 className="font-heading font-bold text-lg text-primary mb-2">{feature.title}</h3>
                  <p className="text-muted-foreground text-sm leading-relaxed">{feature.desc}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Google Reviews - Live Feed */}
      <GoogleReviewsWidget />

      {/* Final CTA */}
      <section className="py-20 bg-white text-center relative overflow-hidden border-t border-gray-200">
        <div className="absolute inset-0 bg-[url('https://www.transparenttextures.com/patterns/cubes.png')] opacity-10"></div>
        <div className="container relative z-10">
          <h2 className="font-heading font-bold text-3xl md:text-5xl mb-6 text-black uppercase italic">Don't Let Plumbing Problems <span className="text-primary">Disrupt Your Day</span></h2>
          <p className="text-xl text-gray-700 max-w-2xl mx-auto mb-10">
            Call the experts at Cotton City Plumbing for fast, reliable service you can trust.
          </p>
          <div className="flex flex-col sm:flex-row justify-center gap-4">
            <a href="tel:8064414110">
              <Button size="lg" className="bg-primary hover:bg-primary/90 text-white font-bold text-lg px-10 py-6 shadow-xl w-full sm:w-auto uppercase rounded-none">
                <Phone className="mr-2 h-5 w-5" /> Call (806) 441-4110
              </Button>
            </a>
            <Link href="/contact">
              <Button size="lg" className="bg-primary hover:bg-primary/90 text-white font-bold text-lg px-10 py-6 w-full sm:w-auto uppercase rounded-none shadow-xl">
                Book Online Now
              </Button>
            </Link>
          </div>
        </div>
      </section>
    </Layout>
  );
}
