import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { trpc } from "@/lib/trpc";
import { Loader2, CheckCircle, Star, Eye, EyeOff, Trash2, MapPin, Calendar, User } from "lucide-react";
import { useState } from "react";
import { toast } from "sonner";

export default function AdminJobPins() {

  const [selectedStatus, setSelectedStatus] = useState<string>("all");
  
  // Fetch all job pins (including pending/hidden)
  const { data: jobPins, isLoading, refetch } = trpc.jobPins.listAll.useQuery();
  
  // Mutations
  const updateStatusMutation = trpc.jobPins.updateStatus.useMutation({
    onSuccess: () => {
      toast.success("Job pin status has been updated successfully.");
      refetch();
    },
    onError: (error) => {
      toast.error(error.message);
    },
  });

  const deleteMutation = trpc.jobPins.delete.useMutation({
    onSuccess: () => {
      toast.success("Job pin has been deleted successfully.");
      refetch();
    },
    onError: (error) => {
      toast.error(error.message);
    },
  });

  const handleStatusChange = (id: number, status: "pending" | "approved" | "featured" | "hidden") => {
    updateStatusMutation.mutate({ id, status });
  };

  const handleDelete = (id: number) => {
    if (confirm("Are you sure you want to delete this job pin? This action cannot be undone.")) {
      deleteMutation.mutate({ id });
    }
  };

  // Filter job pins by selected status
  const filteredPins = jobPins?.filter(pin => 
    selectedStatus === "all" || pin.status === selectedStatus
  );

  // Count by status
  const counts = {
    all: jobPins?.length || 0,
    pending: jobPins?.filter(p => p.status === "pending").length || 0,
    approved: jobPins?.filter(p => p.status === "approved").length || 0,
    featured: jobPins?.filter(p => p.status === "featured").length || 0,
    hidden: jobPins?.filter(p => p.status === "hidden").length || 0,
  };

  const getStatusBadge = (status: string) => {
    const variants: Record<string, { variant: "default" | "secondary" | "destructive" | "outline", label: string }> = {
      pending: { variant: "secondary", label: "Pending" },
      approved: { variant: "default", label: "Approved" },
      featured: { variant: "default", label: "Featured" },
      hidden: { variant: "destructive", label: "Hidden" },
    };
    
    const config = variants[status] || variants.pending;
    return <Badge variant={config.variant}>{config.label}</Badge>;
  };

  return (
    <Layout>
      <SEO 
        title="Admin - Job Pins Moderation | Cotton City Plumbing"
        description="Manage and moderate job pins submitted by technicians."
        canonical="/admin/job-pins"
      />

      <section className="bg-black text-white py-12">
        <div className="container">
          <h1 className="font-league-gothic text-5xl md:text-6xl mb-2 uppercase tracking-wide">Job Pin Moderation</h1>
          <p className="text-xl text-gray-300">Review and manage job pins submitted by technicians</p>
        </div>
      </section>

      <section className="py-8 bg-white">
        <div className="container">
          {/* Status Filter Tabs */}
          <div className="flex flex-wrap gap-2 mb-6">
            {[
              { key: "all", label: "All", count: counts.all },
              { key: "pending", label: "Pending Review", count: counts.pending },
              { key: "approved", label: "Approved", count: counts.approved },
              { key: "featured", label: "Featured", count: counts.featured },
              { key: "hidden", label: "Hidden", count: counts.hidden },
            ].map(({ key, label, count }) => (
              <Button
                key={key}
                variant={selectedStatus === key ? "default" : "outline"}
                onClick={() => setSelectedStatus(key)}
                className="font-bold"
              >
                {label} ({count})
              </Button>
            ))}
          </div>

          {/* Loading State */}
          {isLoading && (
            <div className="flex items-center justify-center py-12">
              <Loader2 className="h-8 w-8 animate-spin text-red-600" />
            </div>
          )}

          {/* Empty State */}
          {!isLoading && filteredPins && filteredPins.length === 0 && (
            <div className="text-center py-12">
              <p className="text-gray-600 font-bold">No job pins found</p>
              <p className="text-gray-500 text-sm mt-2">
                {selectedStatus === "pending" 
                  ? "All job pins have been reviewed" 
                  : "Try changing the filter"}
              </p>
            </div>
          )}

          {/* Job Pins Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredPins && filteredPins.map((job) => (
              <Card key={job.id} className="overflow-hidden">
                <CardHeader className="pb-3">
                  <div className="flex items-start justify-between gap-2">
                    <CardTitle className="text-lg line-clamp-2">{job.title}</CardTitle>
                    {getStatusBadge(job.status)}
                  </div>
                </CardHeader>
                <CardContent className="space-y-3">
                  {/* Photo */}
                  <div className="rounded-lg overflow-hidden h-40 w-full">
                    <img 
                      src={job.photoUrl} 
                      alt={job.title} 
                      className="w-full h-full object-cover"
                      loading="lazy"
                    />
                  </div>

                  {/* Metadata */}
                  <div className="space-y-1 text-sm">
                    {job.technicianName && (
                      <div className="flex items-center gap-2 text-gray-600">
                        <User className="h-4 w-4" />
                        <span>{job.technicianName}</span>
                      </div>
                    )}
                    <div className="flex items-center gap-2 text-gray-600">
                      <Calendar className="h-4 w-4" />
                      <span>{new Date(job.createdAt).toLocaleDateString('en-US', { 
                        month: 'short', 
                        day: 'numeric', 
                        year: 'numeric' 
                      })}</span>
                    </div>
                    <div className="flex items-center gap-2 text-gray-600">
                      <MapPin className="h-4 w-4" />
                      <span className="line-clamp-1">{job.address}</span>
                    </div>
                  </div>

                  {/* Description */}
                  <p className="text-sm text-gray-600 line-clamp-3">{job.description}</p>

                  {/* Review */}
                  {job.customerReview && (
                    <div className="bg-gray-50 p-2 rounded border border-gray-100">
                      {job.rating && (
                        <div className="flex items-center gap-1 mb-1">
                          {[...Array(job.rating)].map((_, i) => (
                            <Star key={i} className="h-3 w-3 fill-yellow-400 text-yellow-400" />
                          ))}
                        </div>
                      )}
                      <p className="text-xs italic text-gray-500 line-clamp-2">"{job.customerReview}"</p>
                    </div>
                  )}

                  {/* Action Buttons */}
                  <div className="grid grid-cols-2 gap-2 pt-2">
                    {job.status !== "approved" && (
                      <Button
                        size="sm"
                        variant="default"
                        onClick={() => handleStatusChange(job.id, "approved")}
                        disabled={updateStatusMutation.isPending}
                        className="bg-green-600 hover:bg-green-700"
                      >
                        <CheckCircle className="h-4 w-4 mr-1" />
                        Approve
                      </Button>
                    )}
                    
                    {job.status !== "featured" && (
                      <Button
                        size="sm"
                        variant="default"
                        onClick={() => handleStatusChange(job.id, "featured")}
                        disabled={updateStatusMutation.isPending}
                        className="bg-blue-600 hover:bg-blue-700"
                      >
                        <Star className="h-4 w-4 mr-1" />
                        Feature
                      </Button>
                    )}

                    {job.status !== "hidden" && (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleStatusChange(job.id, "hidden")}
                        disabled={updateStatusMutation.isPending}
                      >
                        <EyeOff className="h-4 w-4 mr-1" />
                        Hide
                      </Button>
                    )}

                    {job.status === "hidden" && (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleStatusChange(job.id, "approved")}
                        disabled={updateStatusMutation.isPending}
                      >
                        <Eye className="h-4 w-4 mr-1" />
                        Unhide
                      </Button>
                    )}

                    <Button
                      size="sm"
                      variant="destructive"
                      onClick={() => handleDelete(job.id)}
                      disabled={deleteMutation.isPending}
                    >
                      <Trash2 className="h-4 w-4 mr-1" />
                      Delete
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>
    </Layout>
  );
}
