import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import FAQSchema from "@/components/FAQSchema";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from "@/components/ui/accordion";
import { Check, Phone, ChevronRight } from "lucide-react";
import { Link } from "wouter";

interface ServicePageProps {
  title: string;
  subtitle: string;
  heroImage: string;
  introText: string;
  benefits: string[];
  subServices: { title: string; desc: string }[];
  processSteps: { title: string; desc: string }[];
  faqs: { question: string; answer: string }[];
  ctaText?: string;
  children?: React.ReactNode;
}

export default function ServicePageLayout({
  title,
  subtitle,
  heroImage,
  introText,
  benefits,
  subServices,
  processSteps,
  faqs,
  ctaText = "Schedule Service Now",
  children
}: ServicePageProps) {
  const seoTitle = `${title} Services in Lubbock, TX`;
  const seoDesc = `Expert ${title.toLowerCase()} services in Lubbock by Cotton City Plumbing. ${introText.substring(0, 120)}... Call (806) 441-4110 for a free estimate.`;

  return (
    <Layout>
      <SEO 
        title={seoTitle}
        description={seoDesc}
        canonical={`/services/${title.toLowerCase().replace(/\s+/g, '-')}`}
      />
      <FAQSchema faqs={faqs} />
      {/* Hero Section */}
      <section className="relative h-[50vh] min-h-[400px] flex items-center justify-center overflow-hidden bg-primary">
        <div className="absolute inset-0 z-0">
          <img 
            src={heroImage} 
            alt={title} 
            className="w-full h-full object-cover opacity-40"
          />
          <div className="absolute inset-0 bg-gradient-to-r from-primary/90 via-primary/70 to-transparent"></div>
        </div>
        <div className="container relative z-10 text-white">
          <div className="max-w-3xl animate-in slide-in-from-bottom duration-700">
            <h1 className="font-heading font-extrabold text-4xl md:text-6xl mb-4 leading-tight">{title}</h1>
            <p className="text-xl md:text-2xl text-gray-200 font-light mb-8">{subtitle}</p>
            <div className="flex flex-col sm:flex-row gap-4">
              <a href="tel:8064414110">
                <Button size="lg" className="bg-white text-primary hover:bg-gray-100 font-bold text-lg px-8 py-6 shadow-xl w-full sm:w-auto">
                  <Phone className="mr-2 h-5 w-5" /> Call (806) 441-4110
                </Button>
              </a>
              <Link href="/contact">
                <Button size="lg" className="bg-white text-primary hover:bg-gray-100 font-bold text-lg px-8 py-6 w-full sm:w-auto">
                  Book Online
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </section>

      <div className="container py-16">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-16">
            {/* Intro */}
            <section>
              <h2 className="font-heading font-bold text-3xl text-primary mb-6">Professional {title} in Lubbock</h2>
              <p className="text-lg text-muted-foreground leading-relaxed mb-8 whitespace-pre-line">
                {introText}
              </p>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {benefits.map((benefit, index) => (
                  <div key={index} className="flex items-start gap-3 p-4 bg-muted/30 rounded-lg">
                    <div className="bg-secondary/10 p-1 rounded-full mt-0.5">
                      <Check className="h-4 w-4 text-secondary" />
                    </div>
                    <span className="font-medium text-foreground">{benefit}</span>
                  </div>
                ))}
              </div>
            </section>

            {/* Sub-Services or Custom Children */}
            {children ? (
              children
            ) : (
              <section>
                <h3 className="font-heading font-bold text-2xl text-primary mb-6">Our {title} Services</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {subServices.map((service, index) => (
                    <Card key={index} className="border-l-4 border-l-primary shadow-sm hover:shadow-md transition-shadow">
                      <CardContent className="p-6">
                        <h4 className="font-bold text-lg text-primary mb-2">{service.title}</h4>
                        <p className="text-sm text-muted-foreground">{service.desc}</p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </section>
            )}

            {/* Process */}
            <section>
              <h3 className="font-heading font-bold text-2xl text-primary mb-6">Our Process</h3>
              <div className="space-y-6 relative before:absolute before:left-[19px] before:top-2 before:bottom-2 before:w-0.5 before:bg-muted">
                {processSteps.map((step, index) => (
                  <div key={index} className="flex gap-6 relative">
                    <div className="flex-shrink-0 w-10 h-10 rounded-full bg-primary text-white flex items-center justify-center font-bold z-10 border-4 border-white shadow-sm">
                      {index + 1}
                    </div>
                    <div>
                      <h4 className="font-bold text-lg text-primary mb-1">{step.title}</h4>
                      <p className="text-muted-foreground">{step.desc}</p>
                    </div>
                  </div>
                ))}
              </div>
            </section>

            {/* FAQs */}
            <section>
              <h3 className="font-heading font-bold text-2xl text-primary mb-6">Frequently Asked Questions</h3>
              <Accordion type="single" collapsible className="w-full">
                {faqs.map((faq, index) => (
                  <AccordionItem key={index} value={`item-${index}`}>
                    <AccordionTrigger className="text-left font-bold text-foreground hover:text-primary">
                      {faq.question}
                    </AccordionTrigger>
                    <AccordionContent className="text-muted-foreground leading-relaxed">
                      {faq.answer}
                    </AccordionContent>
                  </AccordionItem>
                ))}
              </Accordion>
            </section>
          </div>

          {/* Sidebar */}
          <div className="space-y-8">
            {/* Contact Card */}
            <Card className="bg-primary text-white border-none shadow-xl sticky top-24">
              <CardContent className="p-8 text-center space-y-6">
                <h3 className="font-heading font-bold text-2xl">Need {title}?</h3>
                <p className="text-primary-foreground/80">
                  Contact Cotton City Plumbing today for fast, reliable service in Lubbock and surrounding areas.
                </p>
                <a href="tel:8064414110" className="block">
                  <Button size="lg" className="w-full bg-white text-primary hover:bg-gray-100 font-bold">
                    <Phone className="mr-2 h-4 w-4" /> (806) 441-4110
                  </Button>
                </a>
                <Link href="/contact">
                  <Button className="w-full bg-white text-primary hover:bg-gray-100 font-bold">
                    Request Quote Online
                  </Button>
                </Link>
              </CardContent>
            </Card>

            {/* Other Services Menu */}
            <div className="bg-muted/30 rounded-xl p-6 border border-border">
              <h3 className="font-heading font-bold text-lg mb-4">Other Services</h3>
              <ul className="space-y-2">
                {[
                  { name: "Emergency Plumbing", path: "/services/emergency" },
                  { name: "Water Heaters", path: "/services/water-heaters" },
                  { name: "Drain Cleaning", path: "/services/drain-cleaning" },
                  { name: "Leak Detection", path: "/services/leak-detection" },
                  { name: "Gas Lines", path: "/services/gas-lines" },
                  { name: "Water Treatment", path: "/services/water-treatment" },
                ].map((item) => (
                  <li key={item.path}>
                    <Link href={item.path} className="flex items-center justify-between p-2 rounded hover:bg-white hover:text-primary transition-colors text-muted-foreground group">
                        {item.name}
                        <ChevronRight className="h-4 w-4 opacity-0 group-hover:opacity-100 transition-opacity" />
                    </Link>
                  </li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      </div>

      {/* Bottom CTA */}
      <section className="py-16 bg-primary text-white text-center">
        <div className="container">
          <h2 className="font-heading font-bold text-3xl mb-6">{ctaText}</h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto opacity-90">
            Don't wait until a small problem becomes a major disaster. Call the experts at Cotton City Plumbing today.
          </p>
          <a href="tel:8064414110">
            <Button size="lg" className="bg-white text-primary hover:bg-gray-100 font-bold text-lg px-10 py-6 shadow-xl">
              Call (806) 441-4110
            </Button>
          </a>
        </div>
      </section>
    </Layout>
  );
}
