import { Helmet } from "react-helmet-async";

interface SEOProps {
  title: string;
  description: string;
  keywords?: string;
  canonical?: string;
  type?: "website" | "article";
  name?: string;
  image?: string;
  schema?: Record<string, any>;
}

export default function SEO({ 
  title, 
  description, 
  keywords,
  canonical,
  type = "website",
  name = "Cotton City Plumbing",
  image = "/images/hero-truck.png",
  schema
}: SEOProps) {
  const siteUrl = "https://cottoncityplumbing.com";
  const fullUrl = canonical ? `${siteUrl}${canonical}` : siteUrl;
  const fullImage = image.startsWith("http") ? image : `${siteUrl}${image}`;

  // Default LocalBusiness Schema
  const defaultSchema = {
    "@context": "https://schema.org",
    "@type": "Plumber",
    "name": "Cotton City Plumbing",
    "image": fullImage,
    "url": siteUrl,
    "telephone": "+18064414110",
    "address": {
      "@type": "PostalAddress",
      "addressLocality": "Lubbock",
      "addressRegion": "TX",
      "postalCode": "79424",
      "addressCountry": "US"
    },
    "geo": {
      "@type": "GeoCoordinates",
      "latitude": 33.5779,
      "longitude": -101.8552
    },
    "openingHoursSpecification": [
      {
        "@type": "OpeningHoursSpecification",
        "dayOfWeek": [
          "Monday",
          "Tuesday",
          "Wednesday",
          "Thursday",
          "Friday"
        ],
        "opens": "07:00",
        "closes": "19:00"
      },
      {
        "@type": "OpeningHoursSpecification",
        "dayOfWeek": [
          "Saturday",
          "Sunday"
        ],
        "opens": "00:00",
        "closes": "23:59",
        "description": "Emergency Service Available"
      }
    ],
    "priceRange": "$$",
    "areaServed": [
      {
        "@type": "City",
        "name": "Lubbock"
      },
      {
        "@type": "City",
        "name": "Wolfforth"
      },
      {
        "@type": "City",
        "name": "Slaton"
      },
      {
        "@type": "City",
        "name": "Shallowater"
      }
    ],
    "sameAs": [
      "https://www.facebook.com/cottoncityplumbing",
      "https://www.instagram.com/cottoncityplumbing"
    ]
  };

  const finalSchema = schema || defaultSchema;

  return (
    <Helmet>
      {/* Standard Metadata */}
      <title>{title} | Cotton City Plumbing Lubbock</title>
      <meta name="description" content={description} />
      {keywords && <meta name="keywords" content={keywords} />}
      <link rel="canonical" href={fullUrl} />

      {/* Open Graph / Facebook */}
      <meta property="og:type" content={type} />
      <meta property="og:url" content={fullUrl} />
      <meta property="og:title" content={title} />
      <meta property="og:description" content={description} />
      <meta property="og:image" content={fullImage} />
      <meta property="og:site_name" content={name} />

      {/* Twitter */}
      <meta name="twitter:card" content="summary_large_image" />
      <meta name="twitter:title" content={title} />
      <meta name="twitter:description" content={description} />
      <meta name="twitter:image" content={fullImage} />

      {/* Structured Data (JSON-LD) */}
      <script type="application/ld+json">
        {JSON.stringify(finalSchema)}
      </script>
    </Helmet>
  );
}
