import { Link, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import LocalBusinessSchema from "@/components/LocalBusinessSchema";
import { Phone, Menu, X, Clock, MapPin, Mail, Facebook, Instagram, Link as LinkIcon } from "lucide-react";
import { useState, useEffect } from "react";
import { cn } from "@/lib/utils";

declare global {
  interface Window {
    HCPWidget: any;
  }
}

export default function Layout({ children }: { children: React.ReactNode }) {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [location] = useLocation();

  // Close menu when route changes
  useEffect(() => {
    setIsMenuOpen(false);
  }, [location]);

  // Load Housecall Pro script globally in Layout so it's available for the header button
  useEffect(() => {
    // Check if script already exists to avoid duplicates
    if (!document.querySelector('script[src*="online-booking.housecallpro.com"]')) {
      const script = document.createElement("script");
      script.src = "https://online-booking.housecallpro.com/script.js?token=ae9ce9953f0f4e379d45293c0f23cbb8&orgName=Cotton-City-Plumbing";
      script.async = true;
      document.body.appendChild(script);
    }
  }, []);

  const openBookingModal = () => {
    if (window.HCPWidget) {
      window.HCPWidget.openModal();
    } else {
      // Fallback if widget hasn't loaded yet
      window.location.href = "/contact";
    }
  };

  const navItems = [
    { name: "Home", path: "/" },
    { name: "About", path: "/about" },
    { name: "Services", path: "/services" },
    { name: "Gallery", path: "/gallery" },
    { name: "Recent Jobs", path: "/recent-jobs" },
    { name: "Blog", path: "/blog" },
    { name: "Financing", path: "/financing" },
    { name: "Contact", path: "/contact" },
  ];

  return (
    <>
      <LocalBusinessSchema />
      <div className="min-h-screen flex flex-col font-body">
      {/* Header */}
      <header className="bg-black border-b border-white/10 sticky top-0 z-40 shadow-sm">
        <div className="container py-4">
          <div className="flex items-center justify-between">
            {/* Logo Area */}
            <Link href="/" className="flex items-center gap-3 group">
                <div className="relative h-16 md:h-20 overflow-hidden group-hover:scale-105 transition-transform duration-300">
                  <img 
                    src="/images/logo.png" 
                    alt="Cotton City Plumbing Logo" 
                    className="h-full w-auto object-contain"
                  />
                </div>
            </Link>

            {/* Desktop Navigation */}
            <nav className="hidden lg:flex items-center gap-6">
              {navItems.map((item) => (
                <Link key={item.path} href={item.path} className={cn(
                    "text-2xl font-heading font-bold uppercase tracking-wide transition-colors hover:text-primary relative py-1 after:content-[''] after:absolute after:left-0 after:bottom-0 after:h-0.5 after:w-0 after:bg-primary after:transition-all hover:after:w-full",
                    location === item.path ? "text-primary after:w-full" : "text-white"
                  )}>
                    {item.name}
                </Link>
              ))}
            </nav>

            {/* CTA Buttons */}
            <div className="hidden md:flex items-center gap-4">
              <a href="tel:8064414110" className="flex items-center gap-2 text-white font-heading font-bold text-xl hover:text-primary transition-colors">
                <Phone className="h-6 w-6" />
                <span className="hidden xl:inline">(806) 441-4110</span>
              </a>
              <a href="mailto:services@cottoncityplumbing.com" className="flex items-center gap-2 text-white font-heading font-bold text-xl hover:text-primary transition-colors">
                <Mail className="h-6 w-6" />
                <span className="hidden xl:inline">services@cottoncityplumbing.com</span>
              </a>
              <Button 
                onClick={openBookingModal}
                className="bg-primary hover:bg-primary/90 text-white font-heading font-bold text-xl uppercase px-6 py-6 shadow-lg hover:shadow-xl transition-all transform hover:-translate-y-0.5 rounded-none border-2 border-primary hover:border-white"
              >
                Book Online
              </Button>
            </div>

            {/* Mobile Menu Toggle */}
            <button 
              className="lg:hidden p-2 text-primary"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
              aria-label="Toggle navigation menu"
              aria-expanded={isMenuOpen}
            >
              {isMenuOpen ? <X className="h-8 w-8" /> : <Menu className="h-8 w-8" />}
            </button>
          </div>
        </div>

        {/* Mobile Navigation Menu */}
        {isMenuOpen && (
          <div className="lg:hidden bg-background border-t border-border absolute w-full shadow-xl animate-in slide-in-from-top-5 duration-200">
            <div className="container py-6 flex flex-col gap-4">
              {navItems.map((item) => (
                <Link key={item.path} href={item.path} className={cn(
                    "text-lg font-medium py-2 border-b border-border/50 hover:text-primary hover:pl-2 transition-all",
                    location === item.path ? "text-primary pl-2 border-l-4 border-l-secondary border-b-0" : "text-foreground"
                  )}>
                    {item.name}
                </Link>
              ))}
              <div className="flex flex-col gap-4 mt-4">
                <Button 
                  onClick={() => {
                    setIsMenuOpen(false);
                    openBookingModal();
                  }}
                  className="w-full bg-secondary hover:bg-secondary/90 text-white font-bold py-6 text-lg"
                >
                  Book Online
                </Button>
                <a href="tel:8064414110" className="w-full">
                  <Button variant="outline" className="w-full border-primary text-primary font-bold py-6 text-lg gap-2">
                    <Phone className="h-5 w-5" /> Call (806) 441-4110
                  </Button>
                </a>
              </div>
            </div>
          </div>
        )}
      </header>

      {/* Main Content */}
      <main className="flex-grow">
        {children}
      </main>

      {/* Footer */}
      <footer className="bg-black text-white pt-16 pb-8 border-t-4 border-primary">
        <div className="container">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-8 mb-12">
            {/* Brand Column */}
            <div className="space-y-6">
              <div className="flex flex-col">
                <span className="font-heading font-bold text-3xl text-white leading-none italic">COTTON CITY</span>
                <span className="font-heading font-bold text-2xl text-primary leading-none italic">PLUMBING</span>
              </div>
              <p className="text-gray-400 leading-relaxed">
                Lubbock's trusted, faith-based plumbing experts. We make sure everything runs through Lubbock with honesty, integrity, and excellence.
              </p>
              <div className="flex gap-4">
                <a href="https://www.facebook.com/CottonCityPlumbing/" target="_blank" rel="noopener noreferrer" className="bg-white/10 p-2 rounded-full hover:bg-primary transition-colors">
                  <Facebook className="h-5 w-5" />
                </a>
                <a href="https://www.instagram.com/cottoncityplumbing/" target="_blank" rel="noopener noreferrer" className="bg-white/10 p-2 rounded-full hover:bg-primary transition-colors">
                  <Instagram className="h-5 w-5" />
                </a>
                <a href="#" className="bg-white/10 p-2 rounded-full hover:bg-primary transition-colors">
                  <LinkIcon className="h-5 w-5" />
                </a>
              </div>
            </div>

            {/* Services Column */}
            <div>
              <h3 className="font-heading font-bold text-xl mb-4 text-white border-b border-primary pb-2 inline-block uppercase">Services</h3>
              <ul className="space-y-2">
                {[
                  { name: "Emergency Plumbing", link: "/services/emergency-plumbing" },
                  { name: "Water Heaters", link: "/services/water-heaters" },
                  { name: "Drain Cleaning", link: "/services/drain-cleaning" },
                  { name: "Leak Detection", link: "/services/leak-detection" },
                  { name: "Gas Lines", link: "/services/gas-lines" },
                  { name: "Water Treatment", link: "/services/water-treatment" },
                  { name: "Sewer Repair", link: "/services/sewer-repair" },
                  { name: "Repiping", link: "/services/repiping" },
                  { name: "Commercial", link: "/services/commercial" },
                  { name: "Slab Leaks", link: "/services/slab-leaks" }
                ].map((service) => (
                  <li key={service.name}>
                    <Link href={service.link} className="text-gray-400 hover:text-primary hover:translate-x-1 transition-all inline-block text-sm">
                        {service.name}
                    </Link>
                  </li>
                ))}
              </ul>
            </div>

            {/* Service Areas Column 1 */}
            <div>
              <h3 className="font-heading font-bold text-xl mb-4 text-white border-b border-primary pb-2 inline-block uppercase">Lubbock Areas</h3>
              <ul className="space-y-2">
                {[
                  { name: "Tech Terrace", link: "/tech-terrace" },
                  { name: "Maxey Park", link: "/maxey-park" },
                  { name: "Lakeridge", link: "/lakeridge" },
                  { name: "Shadow Hills", link: "/shadow-hills" },
                  { name: "Coronado", link: "/coronado" },
                  { name: "Monterey", link: "/monterey" },
                  { name: "Frenship", link: "/frenship" },
                  { name: "Westchester", link: "/westchester" },
                  { name: "West Lubbock", link: "/west-lubbock" },
                  { name: "Vintage Township", link: "/vintage-township" },
                  { name: "Quaker Heights", link: "/quaker-heights" },
                  { name: "Remington Park", link: "/remington-park" },
                  { name: "Raintree", link: "/raintree" },
                  { name: "Preston Smith", link: "/preston-smith" },
                  { name: "Kings Park", link: "/kings-park" },
                  { name: "Kelsey Park", link: "/kelsey-park" },
                  { name: "Fox Ridge", link: "/fox-ridge" },
                  { name: "Lubbock-Cooper", link: "/lubbock-cooper" },
                  { name: "South Fork Ranch", link: "/south-fork" },
                  { name: "Davis Park", link: "/davis-park" }
                ].map((area) => (
                  <li key={area.name}>
                    <Link href={area.link} className="text-gray-400 hover:text-primary hover:translate-x-1 transition-all inline-block text-sm">
                        {area.name}
                    </Link>
                  </li>
                ))}
              </ul>
            </div>

            {/* Service Areas Column 2 */}
            <div>
              <h3 className="font-heading font-bold text-xl mb-4 text-white border-b border-primary pb-2 inline-block uppercase">More Areas</h3>
              <ul className="space-y-2">
                {[
                  { name: "The Ridge", link: "/the-ridge" },
                  { name: "Heart of Lubbock", link: "/heart-of-lubbock" },
                  { name: "North Overton", link: "/north-overton" },
                  { name: "South Overton", link: "/south-overton" },
                  { name: "Wheelock & Monterey", link: "/wheelock-monterey" },
                  { name: "Maedgen Area", link: "/maedgen-area" },
                  { name: "Clapp Park", link: "/clapp-park" },
                  { name: "Arnett Benson", link: "/arnett-benson" },
                  { name: "Guadalupe", link: "/guadalupe" },
                  { name: "Melonie Park", link: "/melonie-park" },
                  { name: "Stubbs-Stewart", link: "/stubbs-stewart" },
                  { name: "Westwind", link: "/westwind" },
                  { name: "Northwest Lubbock", link: "/northwest-lubbock" },
                  { name: "Green Acres", link: "/green-acres" },
                  { name: "Carlisle", link: "/carlisle" },
                  { name: "Chatman", link: "/chatman" },
                  { name: "East Lubbock", link: "/east-lubbock" },
                  { name: "Ballenger", link: "/ballenger" },
                  { name: "Bayless Atkins", link: "/bayless-atkins" },
                  { name: "Bowie", link: "/bowie" }
                ].map((area) => (
                  <li key={area.name}>
                    <Link href={area.link} className="text-gray-400 hover:text-primary hover:translate-x-1 transition-all inline-block text-sm">
                        {area.name}
                    </Link>
                  </li>
                ))}
              </ul>
            </div>

            {/* Contact Column */}
            <div>
              <h3 className="font-heading font-bold text-2xl mb-6 text-white border-b border-primary pb-2 inline-block uppercase">Contact Us</h3>
              <address className="space-y-4 not-italic">
                <div className="flex items-start gap-3">
                  <MapPin className="h-5 w-5 text-primary mt-1 flex-shrink-0" />
                  <span className="text-gray-400">
                    <span className="block font-bold text-white uppercase">Cotton City Plumbing</span>
                    1306 146th Street<br/>
                    Lubbock, TX 79423
                  </span>
                </div>
                <div className="flex items-center gap-3">
                  <Phone className="h-5 w-5 text-primary flex-shrink-0" />
                  <a href="tel:8064414110" className="text-gray-400 hover:text-white font-bold">
                    (806) 441-4110
                  </a>
                </div>
                <div className="flex items-center gap-3">
                  <Mail className="h-5 w-5 text-primary flex-shrink-0" />
                  <a href="mailto:services@cottoncityplumbing.com" className="text-gray-400 hover:text-white font-bold">
                    services@cottoncityplumbing.com
                  </a>
                </div>
                <div className="flex items-center gap-3">
                  <Clock className="h-5 w-5 text-primary flex-shrink-0" />
                  <span className="text-gray-400">
                    Mon-Fri: 7am-7pm<br/>
                    Sat: 8am-12pm | Sun: Closed
                  </span>
                </div>
              </address>
            </div>

            {/* Awards/Badges Column */}
            <div>
              <h3 className="font-heading font-bold text-2xl mb-6 text-white border-b border-primary pb-2 inline-block uppercase">Recognized Excellence</h3>
              <div className="flex flex-col gap-4">
                <img 
                  src="/images/community-choice-2025.png" 
                  alt="2025 Official Community's Choice Awards - Best of Lubbock" 
                  className="w-full h-auto max-w-[180px] drop-shadow-lg hover:scale-105 transition-transform duration-300"
                />
                <img 
                  src="/images/nextdoor-fave-2025.jpg" 
                  alt="2025 Nextdoor Neighborhood Faves Winner" 
                  className="w-full h-auto max-w-[180px] drop-shadow-lg hover:scale-105 transition-transform duration-300 rounded-full border-2 border-white/20"
                />
                <img 
                  src="/images/businessrate-best-of-2025.png" 
                  alt="Best of BusinessRate 2025 - Plumber in Lubbock, TX" 
                  className="w-full h-auto max-w-[180px] drop-shadow-lg hover:scale-105 transition-transform duration-300"
                />
                <div className="mt-4">
                  <h4 className="font-heading font-bold text-lg mb-3 text-white uppercase">We Accept</h4>
                  <img 
                    src="/images/payment-methods.png" 
                    alt="We accept Visa, Mastercard, American Express, Discover, Cash, and Checks" 
                    className="w-full h-auto max-w-[200px] opacity-90 hover:opacity-100 transition-opacity"
                  />
                </div>
              </div>
            </div>
          </div>

          <div className="border-t border-white/10 pt-8 flex flex-col md:flex-row justify-between items-center gap-4">
            <p className="text-gray-500 text-sm">
              &copy; {new Date().getFullYear()} Cotton City Plumbing. All rights reserved. <br/>
              Licensed & Insured M-44446 | <Link href="/privacy-policy" className="hover:text-white underline">Privacy Policy</Link> | <Link href="/terms-of-use" className="hover:text-white underline">Terms of Use</Link>
            </p>
          </div>
        </div>
      </footer>
    </div>
    </>
  );
}
