import { Helmet } from "react-helmet-async";

interface JobPinSchemaProps {
  jobPin: {
    id: number;
    title: string;
    description: string;
    serviceType: string;
    address: string;
    latitude: string;
    longitude: string;
    photoUrl: string;
    technicianName?: string | null;
    customerReview?: string | null;
    customerName?: string | null;
    rating?: number | null;
    createdAt: Date | string;
  };
}

/**
 * Generates Schema.org structured data for a job pin.
 * Includes LocalBusiness, Service, and Review schemas for enhanced SEO.
 */
export function JobPinSchema({ jobPin }: JobPinSchemaProps) {
  const dateString = typeof jobPin.createdAt === 'string' 
    ? jobPin.createdAt 
    : jobPin.createdAt.toISOString();

  // LocalBusiness schema for the plumbing company
  const localBusinessSchema = {
    "@context": "https://schema.org",
    "@type": "LocalBusiness",
    "@id": "https://cottoncityplumbing.com/#organization",
    "name": "Cotton City Plumbing",
    "image": "https://cottoncityplumbing.com/images/logo.png",
    "telephone": "+1-806-441-4110",
    "address": {
      "@type": "PostalAddress",
      "streetAddress": jobPin.address.split(',')[0],
      "addressLocality": "Lubbock",
      "addressRegion": "TX",
      "postalCode": "79401",
      "addressCountry": "US"
    },
    "geo": {
      "@type": "GeoCoordinates",
      "latitude": jobPin.latitude,
      "longitude": jobPin.longitude
    },
    "url": "https://cottoncityplumbing.com",
    "priceRange": "$$"
  };

  // Service schema for the specific plumbing service
  const serviceSchema = {
    "@context": "https://schema.org",
    "@type": "Service",
    "serviceType": jobPin.serviceType,
    "provider": {
      "@type": "LocalBusiness",
      "@id": "https://cottoncityplumbing.com/#organization"
    },
    "areaServed": {
      "@type": "City",
      "name": "Lubbock",
      "address": {
        "@type": "PostalAddress",
        "addressRegion": "TX",
        "addressCountry": "US"
      }
    },
    "description": jobPin.description,
    "image": jobPin.photoUrl,
    "url": `https://cottoncityplumbing.com/recent-jobs#job-${jobPin.id}`
  };

  // Review schema if customer review is available
  const reviewSchema = jobPin.customerReview && jobPin.rating ? {
    "@context": "https://schema.org",
    "@type": "Review",
    "itemReviewed": {
      "@type": "Service",
      "name": jobPin.title,
      "provider": {
        "@type": "LocalBusiness",
        "@id": "https://cottoncityplumbing.com/#organization"
      }
    },
    "reviewRating": {
      "@type": "Rating",
      "ratingValue": jobPin.rating,
      "bestRating": 5,
      "worstRating": 1
    },
    "author": {
      "@type": "Person",
      "name": jobPin.customerName || "Anonymous"
    },
    "reviewBody": jobPin.customerReview,
    "datePublished": dateString
  } : null;

  // JobPosting schema to indicate work performed
  const jobPostingSchema = {
    "@context": "https://schema.org",
    "@type": "JobPosting",
    "title": jobPin.title,
    "description": jobPin.description,
    "datePosted": dateString,
    "hiringOrganization": {
      "@type": "LocalBusiness",
      "@id": "https://cottoncityplumbing.com/#organization"
    },
    "jobLocation": {
      "@type": "Place",
      "address": {
        "@type": "PostalAddress",
        "streetAddress": jobPin.address.split(',')[0],
        "addressLocality": jobPin.address.split(',')[1]?.trim() || "Lubbock",
        "addressRegion": "TX",
        "addressCountry": "US"
      },
      "geo": {
        "@type": "GeoCoordinates",
        "latitude": jobPin.latitude,
        "longitude": jobPin.longitude
      }
    }
  };

  const schemas = [
    localBusinessSchema,
    serviceSchema,
    jobPostingSchema,
    ...(reviewSchema ? [reviewSchema] : [])
  ];

  return (
    <Helmet>
      <script type="application/ld+json">
        {JSON.stringify(schemas)}
      </script>
    </Helmet>
  );
}
