import { Star, ExternalLink, Quote } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { trpc } from "@/lib/trpc";

export default function GoogleReviewsWidget() {
  const googleReviewsUrl = "https://share.google/GbJPTQIf41OKMcNJQ";
  
  // Fetch real Google reviews
  const { data, isLoading } = trpc.reviews.getGoogleReviews.useQuery();
  
  const reviews = data?.reviews || [];
  const rating = data?.rating || 0;
  const totalReviews = data?.totalReviews || 0;

  // Generate Review Schema Markup for Google Search Results
  const reviewSchema = {
    "@context": "https://schema.org",
    "@type": "LocalBusiness",
    "name": "Cotton City Plumbing",
    "image": "https://www.cottoncityplumbing.com/images/logo.png",
    "@id": "https://www.cottoncityplumbing.com",
    "url": "https://www.cottoncityplumbing.com",
    "telephone": "+18064414110",
    "address": {
      "@type": "PostalAddress",
      "streetAddress": "1308 146th Street",
      "addressLocality": "Lubbock",
      "addressRegion": "TX",
      "postalCode": "79423",
      "addressCountry": "US"
    },
    "aggregateRating": {
      "@type": "AggregateRating",
      "ratingValue": rating.toString(),
      "bestRating": "5",
      "worstRating": "1",
      "ratingCount": totalReviews.toString()
    },
    "review": reviews.slice(0, 3).map(review => ({
      "@type": "Review",
      "author": {
        "@type": "Person",
        "name": review.authorName
      },
      "reviewRating": {
        "@type": "Rating",
        "ratingValue": review.rating.toString(),
        "bestRating": "5",
        "worstRating": "1"
      },
      "reviewBody": review.text,
      "datePublished": new Date(review.time * 1000).toISOString()
    }))
  };

  return (
    <section className="py-16 bg-muted/30">
      {/* Schema.org JSON-LD for Rich Results */}
      {rating > 0 && totalReviews > 0 && (
        <script
          type="application/ld+json"
          dangerouslySetInnerHTML={{ __html: JSON.stringify(reviewSchema) }}
        />
      )}
      <div className="container">
        <div className="text-center mb-12">
          <div className="flex items-center justify-center gap-2 mb-4">
            <div className="flex gap-1">
              {[...Array(5)].map((_, i) => (
                <Star 
                  key={i} 
                  className={`h-6 w-6 ${
                    i < Math.floor(rating) 
                      ? "fill-yellow-400 text-yellow-400" 
                      : "text-gray-300"
                  }`} 
                />
              ))}
            </div>
            <span className="text-2xl font-bold">{rating.toFixed(1)}</span>
          </div>
          <h2 className="font-heading font-bold text-3xl md:text-4xl text-primary mb-3">
            Trusted by {totalReviews}+ Lubbock Families
          </h2>
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Don't just take our word for it. See what your neighbors are saying about Cotton City Plumbing.
          </p>
        </div>

        {/* Reviews Grid */}
        {isLoading ? (
          <div className="text-center py-12">
            <p className="text-muted-foreground">Loading reviews...</p>
          </div>
        ) : reviews.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-muted-foreground">No reviews available at this time.</p>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            {reviews.slice(0, 3).map((review, index) => (
              <Card key={index} className="bg-white border-gray-200">
                <CardContent className="p-6">
                  <div className="flex items-center gap-1 mb-3">
                    {[...Array(review.rating)].map((_, i) => (
                      <Star key={i} className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                    ))}
                  </div>
                  <Quote className="h-8 w-8 text-primary/20 mb-2" />
                  <p className="text-gray-700 mb-4 line-clamp-4">
                    {review.text}
                  </p>
                  <div className="flex items-center justify-between text-sm">
                    <span className="font-bold text-gray-900">{review.authorName}</span>
                    <span className="text-gray-500">{review.relativeTime}</span>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* CTA to Leave Review */}
        <div className="text-center">
          <Card className="max-w-2xl mx-auto bg-primary text-primary-foreground border-none">
            <CardContent className="p-8">
              <h3 className="font-heading font-bold text-2xl mb-3">
                Had a Great Experience?
              </h3>
              <p className="mb-6 text-primary-foreground/90">
                We'd love to hear from you! Your feedback helps us serve the Lubbock community better and helps your neighbors find trusted plumbing services.
              </p>
              <Button
                size="lg"
                variant="secondary"
                className="font-bold"
                onClick={() => window.open(googleReviewsUrl, '_blank')}
              >
                <ExternalLink className="mr-2 h-5 w-5" />
                Leave Us a Google Review
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
    </section>
  );
}
